﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.IO;

using WorldLab.MathLib;
using WorldLab.GeoDataLib;
using WorldLab.SampleAnalysisLib;

namespace GeoWeiLogRegBasedOnGridLayer
{
    public partial class FrmILRBSWT : Form
    {
        #region private member variable

        /// <summary>
        /// independent layer list
        /// </summary>
        private List<Grid> independentLayers;

        /// <summary>
        /// dependent layer
        /// </summary>
        private Grid dependentLayer;

        /// <summary>
        /// global weight layer
        /// </summary>
        private Grid globalWeightLayer;

        /// <summary>
        ///  grid rows
        /// </summary>
        private int gridRows;

        /// <summary>
        /// grid columns
        /// </summary>
        private int gridColumns;

        /// <summary>
        /// NoData value
        /// </summary>
        private double gridNoData;


        #endregion

        #region Construction Function

        /// <summary>
        /// Construction
        /// </summary>
        public FrmILRBSWT()
        {
            InitializeComponent();

            comBoxWindow.Items.Add("Square");
            comBoxWindow.Items.Add("Rectangle");
            comBoxWindow.Items.Add("Circle");
            comBoxWindow.Items.Add("Ellipse");
            comBoxWindow.SelectedIndex = 3;
            txtBoxWndSet2.Text = "0.5";// ratio
            txtBoxWndSet6.Text = "1.96";// t

            comBoxKernelFunction.Items.Add("Gaussian Function");
            comBoxKernelFunction.Items.Add("Exponential Function");
            comBoxKernelFunction.SelectedIndex = 1;

            txtBoxOutputDirectory.Text = System.Environment.CurrentDirectory + "\\Data";
        }

        #endregion

        #region Form Event Function

        /// <summary>
        /// load independent layers
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnLoadIndependentLayers_Click(object sender, EventArgs e)
        {
            OpenFileDialog dlg = new OpenFileDialog();
            dlg.Multiselect = true;
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                #region clear loaded independent layers

                listBoxIndependentLayers.Items.Clear();

                if (listBoxDependentLayer.Items.Count == 0 &&
                    listBoxGlobalWeightLayer.Items.Count == 0)
                {
                    labelRows.Text = "";
                    labelColumns.Text = "";
                    labelOriginalX.Text = "";
                    labelOriginalY.Text = "";
                    labelSize.Text = "";
                    labelNodataValue.Text = "";
                }

                if (independentLayers != null)
                {
                    if (independentLayers.Count > 0)
                    {
                        for (int i = 0; i < independentLayers.Count; ++i)
                        {
                            independentLayers[i].Dispose();
                            independentLayers[i] = null;
                        }
                        independentLayers.Clear();
                    }
                    independentLayers = null;
                }

                #endregion

                #region load independent layer

                independentLayers = new List<Grid>();
                Grid indepLayer = null;

                string errorStr = "";

                for (int i = 0; i < dlg.FileNames.Length; ++i)
                {
                    indepLayer = Grid.GetFromTxt(dlg.FileNames[i]);
                    if (indepLayer == null)
                    {
                        errorStr += dlg.FileNames[i];
                        errorStr += "\n";
                        continue;
                    }
                    independentLayers.Add(indepLayer);
                }

                if (labelRows.Text == "")
                {
                    labelRows.Text = independentLayers[0].RowCounts.ToString();
                    labelColumns.Text = independentLayers[0].ColCounts.ToString();
                    labelOriginalX.Text = independentLayers[0].OriginalPosX.ToString();
                    labelOriginalY.Text = independentLayers[0].OriginalPosY.ToString();
                    labelSize.Text = independentLayers[0].Size.ToString();
                    labelNodataValue.Text = independentLayers[0].NoDataValue.ToString();
                }

                for (int i = 0, j = 0; i < independentLayers.Count; ++j)
                {
                    if (independentLayers[i].RowCounts != int.Parse(labelRows.Text) ||
                        independentLayers[i].ColCounts != int.Parse(labelColumns.Text) ||
                        independentLayers[i].OriginalPosX != double.Parse(labelOriginalX.Text) ||
                        independentLayers[i].OriginalPosY != double.Parse(labelOriginalY.Text) ||
                        independentLayers[i].Size != double.Parse(labelSize.Text) ||
                        independentLayers[i].NoDataValue != double.Parse(labelNodataValue.Text))
                    {
                        errorStr += dlg.FileNames[j];
                        errorStr += "\n";
                        independentLayers[i].Dispose();
                        independentLayers[i] = null;
                        independentLayers.RemoveAt(i);
                    }
                    else
                    {
                        listBoxIndependentLayers.Items.Add(dlg.FileNames[j]);
                        ++i;
                    }
                }

                if (errorStr != "")
                {
                    errorStr += "Load Fail.";
                    MessageBox.Show(errorStr);
                }

                #endregion
            }
        }

        /// <summary>
        /// load dependent layer
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnLoadDependentLayer_Click(object sender, EventArgs e)
        {
            OpenFileDialog dlg = new OpenFileDialog();
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                #region clear loaded dependent layer

                listBoxDependentLayer.Items.Clear();
                if (listBoxIndependentLayers.Items.Count == 0 &&
                    listBoxGlobalWeightLayer.Items.Count == 0)
                {
                    labelRows.Text = "";
                    labelColumns.Text = "";
                    labelOriginalX.Text = "";
                    labelOriginalY.Text = "";
                    labelSize.Text = "";
                    labelNodataValue.Text = "";
                }
                if (dependentLayer != null)
                {
                    dependentLayer.Dispose();
                    dependentLayer = null;
                }

                #endregion

                #region load dependent layer

                dependentLayer = Grid.GetFromTxt(dlg.FileName);
                if (dependentLayer == null)
                {
                    string errorStr = dlg.FileName;
                    errorStr += "\n";
                    errorStr += "Load Fail.";
                    MessageBox.Show(errorStr);
                    return;
                }

                if (labelRows.Text == "")
                {
                    labelRows.Text = dependentLayer.RowCounts.ToString();
                    labelColumns.Text = dependentLayer.ColCounts.ToString();
                    labelOriginalX.Text = dependentLayer.OriginalPosX.ToString();
                    labelOriginalY.Text = dependentLayer.OriginalPosY.ToString();
                    labelSize.Text = dependentLayer.Size.ToString();
                    labelNodataValue.Text = dependentLayer.NoDataValue.ToString();

                    listBoxDependentLayer.Items.Add(dlg.FileName);
                }
                else
                {
                    if (dependentLayer.RowCounts != int.Parse(labelRows.Text) ||
                        dependentLayer.ColCounts != int.Parse(labelColumns.Text) ||
                        dependentLayer.OriginalPosX != double.Parse(labelOriginalX.Text) ||
                        dependentLayer.OriginalPosY != double.Parse(labelOriginalY.Text) ||
                        dependentLayer.Size != double.Parse(labelSize.Text) ||
                        dependentLayer.NoDataValue != double.Parse(labelNodataValue.Text))
                    {
                        string errorStr = dlg.FileName;
                        errorStr += "\n";
                        errorStr += "Load Fail.";
                        MessageBox.Show(errorStr);
                    }
                    else
                    {
                        listBoxDependentLayer.Items.Add(dlg.FileName);
                    }
                }

                #endregion

            }
        }

        /// <summary>
        /// load global weight layer
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnLoadGlobalWeightLayer_Click(object sender, EventArgs e)
        {
            OpenFileDialog dlg = new OpenFileDialog();
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                #region clear loaded global weight layer

                listBoxGlobalWeightLayer.Items.Clear();
                if (listBoxIndependentLayers.Items.Count == 0 &&
                    listBoxDependentLayer.Items.Count == 0)
                {
                    labelRows.Text = "";
                    labelColumns.Text = "";
                    labelOriginalX.Text = "";
                    labelOriginalY.Text = "";
                    labelSize.Text = "";
                    labelNodataValue.Text = "";
                }
                if (globalWeightLayer != null)
                {
                    globalWeightLayer.Dispose();
                    globalWeightLayer = null;
                }

                #endregion

                #region load global weight layer

                globalWeightLayer = Grid.GetFromTxt(dlg.FileName);
                if (globalWeightLayer == null)
                {
                    string errorStr = dlg.FileName;
                    errorStr += "\n";
                    errorStr += "Load Fail.";
                    MessageBox.Show(errorStr);
                    return;
                }

                if (labelRows.Text == "")
                {
                    labelRows.Text = globalWeightLayer.RowCounts.ToString();
                    labelColumns.Text = globalWeightLayer.ColCounts.ToString();
                    labelOriginalX.Text = globalWeightLayer.OriginalPosX.ToString();
                    labelOriginalY.Text = globalWeightLayer.OriginalPosY.ToString();
                    labelSize.Text = globalWeightLayer.Size.ToString();
                    labelNodataValue.Text = globalWeightLayer.NoDataValue.ToString();

                    listBoxGlobalWeightLayer.Items.Add(dlg.FileName);
                }
                else
                {
                    if (globalWeightLayer.RowCounts != int.Parse(labelRows.Text) ||
                        globalWeightLayer.ColCounts != int.Parse(labelColumns.Text) ||
                        globalWeightLayer.OriginalPosX != double.Parse(labelOriginalX.Text) ||
                        globalWeightLayer.OriginalPosY != double.Parse(labelOriginalY.Text) ||
                        globalWeightLayer.Size != double.Parse(labelSize.Text) ||
                        globalWeightLayer.NoDataValue != double.Parse(labelNodataValue.Text))
                    {
                        string errorStr = dlg.FileName;
                        errorStr += "\n";
                        errorStr += "Load Fail.";
                        MessageBox.Show(errorStr);
                    }
                    else
                    {
                        listBoxGlobalWeightLayer.Items.Add(dlg.FileName);
                    }
                }

                #endregion
            }
        }

        /// <summary>
        /// define sample search window
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void comBoxSelectWindow_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (comBoxWindow.SelectedIndex)
            {
                case 0:
                    {
                        #region Square

                        labelWndSet0.Show();
                        txtBoxWndSet0.Show();
                        labelWndSet0.Text = "Initial Length of Side";
                        txtBoxWndSet0.Text = "30";

                        labelWndSet1.Show();
                        txtBoxWndSet1.Show();
                        labelWndSet1.Text = "Largest Length of Side";
                        txtBoxWndSet1.Text = "50";

                        labelWndSet3.Hide();
                        txtBoxWndSet3.Hide();

                        labelWndSet4.Hide();
                        txtBoxWndSet4.Hide();

                        labelWndSet5.Hide();
                        txtBoxWndSet5.Hide();

                        #endregion
                    }
                    break;
                case 1:
                    {
                        #region Rectangle

                        labelWndSet0.Show();
                        txtBoxWndSet0.Show();
                        labelWndSet0.Text = "Initial Length of Side";
                        txtBoxWndSet0.Text = "30";

                        labelWndSet1.Show();
                        txtBoxWndSet1.Show();
                        labelWndSet1.Text = "Largest Length of Side";
                        txtBoxWndSet1.Text = "50";

                        labelWndSet3.Show();
                        txtBoxWndSet3.Show();
                        labelWndSet3.Text = "Ratio of side";
                        txtBoxWndSet3.Text = "0.5";

                        labelWndSet4.Hide();
                        txtBoxWndSet4.Hide();

                        labelWndSet5.Hide();
                        txtBoxWndSet5.Hide();

                        #endregion
                    }
                    break;
                case 2:
                    {
                        #region Circle

                        labelWndSet0.Show();
                        txtBoxWndSet0.Show();
                        labelWndSet0.Text = "Initial Radius";
                        txtBoxWndSet0.Text = "30";

                        labelWndSet1.Show();
                        txtBoxWndSet1.Show();
                        labelWndSet1.Text = "Largest Radius";
                        txtBoxWndSet1.Text = "50";

                        labelWndSet3.Hide();
                        txtBoxWndSet3.Hide();

                        labelWndSet4.Hide();
                        txtBoxWndSet4.Hide();

                        labelWndSet5.Hide();
                        txtBoxWndSet5.Hide();

                        #endregion
                    }
                    break;
                case 3:
                    {
                        #region Ellipse

                        labelWndSet0.Show();
                        txtBoxWndSet0.Show();
                        labelWndSet0.Text = "Initial Major Radius ";
                        txtBoxWndSet0.Text = "87";

                        labelWndSet1.Show();
                        txtBoxWndSet1.Show();
                        labelWndSet1.Text = "Largest Major Radius";
                        txtBoxWndSet1.Text = "107";

                        labelWndSet3.Show();
                        txtBoxWndSet3.Show();
                        labelWndSet3.Text = "Ratio of Radius";
                        txtBoxWndSet3.Text = "0.3";

                        labelWndSet4.Show();
                        txtBoxWndSet4.Show();
                        labelWndSet4.Text = "Orientation";
                        txtBoxWndSet4.Text = "65";

                        labelWndSet5.Show();
                        txtBoxWndSet5.Show();
                        labelWndSet5.Text = "Compensation Coefficient for Sill";
                        txtBoxWndSet5.Text = "0.77";

                        #endregion
                    }
                    break;
            }
        }
        /// <summary>
        /// define kernel function
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void comBoxKernelFunction_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (comBoxKernelFunction.SelectedIndex)
            {
                case 0:
                    {
                        #region Gaussian Function

                        labelFuncSet0.Show();
                        txtBoxFuncSet0.Show();
                        labelFuncSet0.Text = "Decay Coefficient";
                        txtBoxFuncSet0.Text = "0.001";

                        string[] temp = AppDomain.CurrentDomain.BaseDirectory.Split("\\".ToCharArray());
                        string respath = "";
                        for (int i = 0; i < temp.Length - 3; i++)
                        {
                            respath += temp[i];
                            respath += "\\";
                        }
                        picFunctionFormula.ImageLocation = respath + "Resources\\GaussianFunction.jpg";

                        #endregion
                    }
                    break;
                case 1:
                    {
                        #region Exponential Function

                        labelFuncSet0.Show();
                        txtBoxFuncSet0.Show();
                        labelFuncSet0.Text = "b";
                        txtBoxFuncSet0.Text = "0.02245728";

                        string[] temp = AppDomain.CurrentDomain.BaseDirectory.Split("\\".ToCharArray());
                        string respath = "";
                        for (int i = 0; i < temp.Length - 3; i++)
                        {
                            respath += temp[i];
                            respath += "\\";
                        }
                        picFunctionFormula.ImageLocation = respath + "Resources\\ExponentialFunction.jpg";

                        #endregion
                    }
                    break;
            }
        }

        /// <summary>
        /// set output directory
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnSetOutputDir_Click(object sender, EventArgs e)
        {
            FolderBrowserDialog dlg = new FolderBrowserDialog();
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                txtBoxOutputDirectory.Text = dlg.SelectedPath;
            }
        }

        /// <summary>
        /// run
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Run_Click(object sender, EventArgs e)
        {
            if (listBoxIndependentLayers.Items.Count == 0 ||
                listBoxDependentLayer.Items.Count == 0)
            {
                MessageBox.Show("Please Load Grid Layers.");
                return;
            }

            List<RectWndTemplate> templeList = DefineTemplates();
            if (templeList.Count == 0)
            {
                MessageBox.Show("Please Redefine Local Window.");
                return;
            }

            if (listBoxGlobalWeightLayer.Items.Count == 0)
            {
                globalWeightLayer = new Grid(dependentLayer);
                for (int i = 0; i < globalWeightLayer.RowCounts; ++i)
                {
                    for (int j = 0; j < globalWeightLayer.ColCounts; ++j)
                    {
                        globalWeightLayer.SetValue(i, j, 1.0);
                    }
                }
            }

            string dir = txtBoxOutputDirectory.Text;
            if (!Directory.Exists(dir))
            {
                Directory.CreateDirectory(dir);
            }

            GridVarLayersAnalysis ana = new GridVarLayersAnalysis(
                independentLayers, dependentLayer, globalWeightLayer,
                templeList, double.Parse(txtBoxWndSet6.Text));

            ComputePI(ana, txtBoxOutputDirectory.Text);
        }

        #endregion

        #region Template Function

        /// <summary>
        /// define temple list for seatch window
        /// </summary>
        /// <returns>temple list</returns>
        private List<RectWndTemplate> DefineTemplates()
        {
            List<RectWndTemplate> templeList = new List<RectWndTemplate>();
            RectWndTemplate template;

            int func = 0;
            double coeff = 0.001;
            switch (comBoxKernelFunction.SelectedIndex)
            {
                case 0:
                    {
                        func = 0;
                        coeff = double.Parse(txtBoxFuncSet0.Text);// Lamda
                    }
                    break;
                case 1:
                    {
                        func = 1;
                        coeff = double.Parse(txtBoxFuncSet0.Text);// dmax
                    }
                    break;

            }

            switch (comBoxWindow.SelectedIndex)
            {
                case 0:
                    {
                        #region Square
                        int iniNums = int.Parse(txtBoxWndSet0.Text);
                        int maxNums = int.Parse(txtBoxWndSet1.Text);
                        double rate = double.Parse(txtBoxWndSet2.Text);
                        int curNums = iniNums;
                        int lastNums;
                        while (curNums <= maxNums)
                        {
                            template = DefineTemplateOnRect(curNums, curNums, func, coeff);
                            templeList.Add(template);
                            lastNums = curNums;
                            curNums = (int)(lastNums + (maxNums - iniNums) * rate);
                            if (curNums == lastNums)
                            {
                                break;
                            }
                            if (curNums > maxNums && lastNums < maxNums)
                            {
                                curNums = maxNums;
                            }
                        }
                        #endregion
                    }
                    break;
                case 1:
                    {
                        #region Rectangle
                        int iniColNums = int.Parse(txtBoxWndSet0.Text);
                        int maxColNums = int.Parse(txtBoxWndSet1.Text);
                        double rate = double.Parse(txtBoxWndSet2.Text);
                        double ratio = double.Parse(txtBoxWndSet3.Text);
                        int curColNums = iniColNums;
                        int lastColNums;
                        while (curColNums <= maxColNums)
                        {
                            template = DefineTemplateOnRect(curColNums, (int)(ratio * curColNums),
                                func, coeff);
                            templeList.Add(template);
                            lastColNums = curColNums;
                            curColNums = (int)(lastColNums + (maxColNums - iniColNums) * rate);
                            if (curColNums == lastColNums)
                            {
                                break;
                            }
                            if (curColNums > maxColNums && lastColNums < maxColNums)
                            {
                                curColNums = maxColNums;
                            }
                        }
                        #endregion
                    }
                    break;
                case 2:
                    {
                        #region Circle
                        int iniRadius = int.Parse(txtBoxWndSet0.Text);
                        int maxRadius = int.Parse(txtBoxWndSet1.Text);
                        double rate = double.Parse(txtBoxWndSet2.Text);
                        int curRadius = iniRadius;
                        int lastRadius;
                        while (curRadius <= maxRadius)
                        {
                            template = DefineTemplateOnCircle(curRadius, func, coeff);
                            templeList.Add(template);
                            lastRadius = curRadius;
                            curRadius = (int)(lastRadius + (maxRadius - iniRadius) * rate);
                            if (curRadius == lastRadius)
                            {
                                break;
                            }
                            if (curRadius > maxRadius && lastRadius < maxRadius)
                            {
                                curRadius = maxRadius;
                            }
                        }
                        #endregion
                    }
                    break;
                case 3:
                    {
                        #region Ellipse
                        int iniMajRadius = int.Parse(txtBoxWndSet0.Text);
                        int maxMajRadius = int.Parse(txtBoxWndSet1.Text);
                        double rate = double.Parse(txtBoxWndSet2.Text);
                        double ratio = double.Parse(txtBoxWndSet3.Text);
                        int curMajRadius = iniMajRadius;
                        int lastMajRadius;
                        Angle ori = new Angle(180 - double.Parse(txtBoxWndSet4.Text));
                        double sill = double.Parse(txtBoxWndSet5.Text);
                        while (curMajRadius <= maxMajRadius)
                        {
                            template = DefineTemplateOnEllipse(
                                     curMajRadius, (int)(ratio * curMajRadius), ori, sill,
                                     func, coeff);
                            templeList.Add(template);
                            lastMajRadius = curMajRadius;
                            curMajRadius = (int)(lastMajRadius +
                                (maxMajRadius - iniMajRadius) * rate);
                            if (curMajRadius == lastMajRadius)
                            {
                                break;
                            }
                            if (curMajRadius > maxMajRadius && lastMajRadius < maxMajRadius)
                            {
                                curMajRadius = maxMajRadius;
                            }
                        }
                        #endregion

                    }
                    break;
            }

            return templeList;
        }

        /// <summary>
        /// define temple list for rectangle seatch window
        /// </summary>
        /// <param name="colNums">the number of cells in the grid row</param>
        /// <param name="rowNums">the number of cells in the grid column</param>
        /// <param name="func">kernel function</param>
        /// <param name="coeff">coefficient</param>
        /// <returns>template</returns>
        private RectWndTemplate DefineTemplateOnRect(int colNums, int rowNums,
            int func, double coeff)
        {
            RectWndTemplate template = new RectWndTemplate(rowNums, colNums);
            TemplateCell cell;
            for (int i = 0; i < rowNums; ++i)
            {
                for (int j = 0; j < colNums; ++j)
                {
                    cell = new TemplateCell();
                    cell.Row = i;
                    cell.Col = j;
                    cell.DisToCentre = Math.Sqrt(
                        (template.CentreCol - j) * (template.CentreCol - j) +
                        (template.CentreRow - i) * (template.CentreRow - i));
                    cell.IsInWindow = true;
                    switch (func)
                    {
                        case 0:
                            {
                                #region Gaussian Function
                                cell.Influence = Math.Exp(
                                    -coeff * cell.DisToCentre * cell.DisToCentre);
                                #endregion
                            }
                            break;
                        case 1:
                            {
                                #region Exponential Function
                                cell.Influence = Math.Exp(
                                    -coeff * cell.DisToCentre * cell.DisToCentre);
                                #endregion
                            }
                            break;
                    }

                    template.SetElement(i, j, cell);
                }
            }
            return template;
        }

        /// <summary>
        /// define temple list for circle seatch window
        /// </summary>
        /// <param name="radius">radius</param>
        /// <param name="func">kernel function</param>
        /// <param name="coeff">coefficient</param>
        /// <returns>template</returns>
        private RectWndTemplate DefineTemplateOnCircle(int radius, int func, double coeff)
        {
            int nums = radius * 2 + 1;
            RectWndTemplate template = new RectWndTemplate(nums, nums);
            TemplateCell cell;
            for (int i = 0; i < nums; ++i)
            {
                for (int j = 0; j < nums; ++j)
                {
                    cell = new TemplateCell();
                    cell.Row = i;
                    cell.Col = j;
                    cell.DisToCentre = Math.Sqrt(
                        (template.CentreCol - j) * (template.CentreCol - j) +
                        (template.CentreRow - i) * (template.CentreRow - i));
                    if (cell.DisToCentre <= radius)
                    {
                        cell.IsInWindow = true;
                        switch (func)
                        {
                            case 0:
                                {
                                    #region Gaussian Function
                                    cell.Influence = Math.Exp(
                                        -coeff * cell.DisToCentre * cell.DisToCentre);
                                    #endregion
                                }
                                break;
                            case 1:
                                {
                                    #region Exponential Function
                                    cell.Influence = Math.Exp(
                                        -coeff * cell.DisToCentre * cell.DisToCentre);
                                    #endregion
                                }
                                break;
                        }
                    }
                    else
                    {
                        cell.IsInWindow = false;
                        cell.Influence = double.NaN;
                    }
                    template.SetElement(i, j, cell);
                }
            }
            return template;
        }

        /// <summary>
        /// define temple list for ellipse seatch window
        /// </summary>
        /// <param name="majRadius">semi-major radius</param>
        /// <param name="minRadius">semi-minor axis</param>
        /// <param name="ori">azimuth</param>
        /// <param name="sill">axial length ratio of influencing ellipse</param>
        /// <param name="func">kernel function</param>
        /// <param name="coeff">coefficient</param>
        /// <returns>template</returns>
        private RectWndTemplate DefineTemplateOnEllipse(
            int majRadius, int minRadius, Angle ori, double sill, int func, double coeff)
        {
            int nums = majRadius * 2 + 1;
            RectWndTemplate template = new RectWndTemplate(nums, nums);
            TemplateCell cell;

            double d = Math.Sqrt(majRadius * majRadius - minRadius * minRadius);
            double ax = d * ori.Sin + template.CentreCol;
            double ay = d * ori.Cos + template.CentreRow;
            double bx = -d * ori.Sin + template.CentreCol;
            double by = -d * ori.Cos + template.CentreRow;

            double pa, pb;
            double a;
            for (int i = 0; i < nums; ++i)
            {
                for (int j = 0; j < nums; ++j)
                {
                    cell = new TemplateCell();
                    cell.Row = i;
                    cell.Col = j;
                    pa = Math.Sqrt((j - ax) * (j - ax) + (i - ay) * (i - ay));
                    pb = Math.Sqrt((j - bx) * (j - bx) + (i - by) * (i - by));
                    if (pa + pb <= 2 * majRadius)
                    {
                        cell.IsInWindow = true;

                        switch (func)
                        {
                            case 0:
                                {
                                    #region Gaussian Function
                                    cell.Influence = Math.Exp(
                                        -coeff * cell.DisToCentre * cell.DisToCentre);
                                    #endregion
                                }
                                break;
                            case 1:
                                {
                                    #region Exponential Function
                                    a = Math.Sqrt(
                                            Math.Pow((j - template.CentreCol) * ori.Sin +
                                                     (i - template.CentreRow) * ori.Cos, 2) +
                                            Math.Pow((j - template.CentreCol) * ori.Cos -
                                                     (i - template.CentreRow) * ori.Sin, 2) /
                                            Math.Pow(sill, 2)
                                                 );
                                    cell.Influence = 1 - (Math.Exp(coeff * a) - 1) /
                                                         (Math.Exp(coeff * majRadius) - 1);
                                    #endregion
                                }
                                break;
                        }

                    }
                    else
                    {
                        cell.IsInWindow = false;
                        cell.Influence = double.NaN;
                    }

                    template.SetElement(i, j, cell);
                }
            }
            return template;
        }

        #endregion

        /// <summary>
        /// compute PI of each cell, and write result into file
        /// </summary>
        /// <param name="ana">grid analysis</param>
        /// <param name="dir">directory</param>
        private void ComputePI(GridVarLayersAnalysis ana, string dir)
        {
            int indepCount = listBoxIndependentLayers.Items.Count;

            string piFileName = dir + "\\Pi_" + txtBoxWndSet0.Text + ".txt";
            string[] betaFileName = new string[indepCount + 1];
            string tFileName = dir + "\\Pi_" + txtBoxWndSet0.Text + "_t" + ".txt";

            FileStream[] betaFSW = new FileStream[indepCount + 1];
            StreamWriter[] betaSW = new StreamWriter[indepCount + 1];
            for (int i = 0; i < indepCount + 1; ++i)
            {
                betaFSW[i] = new FileStream(dir + "\\Pi_" + txtBoxWndSet0.Text + "_b" + i.ToString()
                    + ".txt", FileMode.Create);
                betaSW[i] = new StreamWriter(betaFSW[i]);
            }

            FileStream fswt = new FileStream(tFileName, FileMode.Create);
            StreamWriter swt = new StreamWriter(fswt);
            FileStream fsw = new FileStream(piFileName, FileMode.Create);
            StreamWriter sw = new StreamWriter(fsw);

            #region write grid information into PI file

            string fileNameR = listBoxIndependentLayers.Items[0].ToString();
            FileStream fsr = new FileStream(fileNameR, FileMode.Open, FileAccess.Read);
            StreamReader sr = new StreamReader(fsr);
            string lineText = "";
            int lineIndex = 0;
            while ((lineText = sr.ReadLine()) != null && lineIndex < 6)
            {
                sw.WriteLine(lineText);
                for (int i = 0; i < indepCount + 1; ++i)
                {
                    betaSW[i].WriteLine(lineText);
                }
                swt.WriteLine(lineText);
                lineIndex++;
            }

            sr.Close();
            sr.Dispose();
            sr = null;
            fsr.Close();
            fsr.Dispose();
            fsr = null;

            #endregion

            int grdRows = int.Parse(labelRows.Text);
            int grdCols = int.Parse(labelColumns.Text);

            List<int> noValCovIdxList = null;
            double[] beta = null;
            double t = double.NaN;
            double pi;
            string piLineText;
            string[] betaLineText;
            string tLineText;
            for (int i = 0; i < grdRows; ++i)
            {
                piLineText = "";
                betaLineText = new string[indepCount + 1];
                for (int j = 0; j < indepCount + 1; ++j)
                {
                    betaLineText[j] = "";
                }
                tLineText = "";
                for (int j = 0; j < grdCols; ++j)
                {
                    noValCovIdxList = null;
                    beta = null;
                    t = double.NaN;
                    pi = ana.ComputePI(j, i, ref noValCovIdxList, ref beta, ref t);

                    piLineText += pi.ToString();
                    if (double.IsNaN(t))
                    {
                        tLineText += "-9999";
                    }
                    else
                    {
                        tLineText += t.ToString();
                    }

                    if (noValCovIdxList == null || noValCovIdxList.Count == indepCount)
                    {
                        for (int k = 0; k < indepCount + 1; ++k)
                        {
                            betaLineText[k] += "-9999";
                        }
                    }
                    else
                    {
                        betaLineText[0] += beta[0].ToString();

                        List<int> valCovIdxList = new List<int>();
                        for (int k = 0; k < indepCount; ++k)
                        {
                            if (!noValCovIdxList.Contains(k))
                            {
                                valCovIdxList.Add(k);
                            }
                        }
                        for (int k = 1; k < indepCount + 1; ++k)
                        {
                            if (noValCovIdxList.Contains(k - 1))
                            {
                                betaLineText[k] += "-9999";
                            }
                            else
                            {
                                int xx = valCovIdxList.IndexOf(k - 1);
                                betaLineText[k] += beta[xx + 1].ToString();
                            }
                        }
                        valCovIdxList.Clear();
                    }

                    if (j < grdCols - 1)
                    {
                        piLineText += " ";
                        for (int k = 0; k < indepCount + 1; ++k)
                        {
                            betaLineText[k] += " ";
                        }
                        tLineText += " ";
                    }
                }

                sw.WriteLine(piLineText);
                for (int k = 0; k < indepCount + 1; ++k)
                {
                    betaSW[k].WriteLine(betaLineText[k]);
                }
                swt.WriteLine(tLineText);
            }
            sw.Close();
            sw.Dispose();
            sw = null;
            fsw.Close();
            fsw.Dispose();
            fsw = null;
            for (int k = 0; k < indepCount + 1; ++k)
            {
                betaSW[k].Close();
                betaSW[k].Dispose();
                betaSW[k] = null;
                betaFSW[k].Close();
                betaFSW[k].Dispose();
                betaFSW[k] = null;

            }
            swt.Close();
            swt.Dispose();
            swt = null;
            fswt.Close();
            fswt.Dispose();
            fswt = null;
            GC.Collect();
        }

    }
}

